/* 
* Bcmchar.c
*
*Copyright (C) 2010 Beceem Communications, Inc.
*
*This program is free software: you can redistribute it and/or modify 
*it under the terms of the GNU General Public License version 2 as
*published by the Free Software Foundation. 
*
*This program is distributed in the hope that it will be useful,but 
*WITHOUT ANY WARRANTY; without even the implied warranty of
*MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
*See the GNU General Public License for more details.
*
*You should have received a copy of the GNU General Public License
*along with this program. If not, write to the Free Software Foundation, Inc.,
*51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
*/


#include <headers.h>
/***************************************************************
* Function	  - bcm_char_open()
*
* Description - This is the "open" entry point for the character
*				driver.
*
* Parameters  - inode: Pointer to the Inode structure of char device 
*				filp : File pointer of the char device
*
* Returns	  - Zero(Success)
****************************************************************/
static struct class *bcm_class = NULL;
static int bcm_char_open(struct inode *inode, struct file * filp)
{
	PMINI_ADAPTER 		Adapter = NULL;
    PPER_TARANG_DATA 	pTarang = NULL;
	
	Adapter = GET_BCM_ADAPTER(gblpnetdev);
    pTarang = (PPER_TARANG_DATA)kmalloc(sizeof(PER_TARANG_DATA), GFP_KERNEL);
    if (!pTarang)
        return -ENOMEM; 
	
	memset (pTarang, 0, sizeof(PER_TARANG_DATA));
    pTarang->Adapter = Adapter;
	pTarang->RxCntrlMsgBitMask = 0xFFFFFFFF & ~(1 << 0xB) ;

	down(&Adapter->RxAppControlQueuelock);
    pTarang->next = Adapter->pTarangs;
    Adapter->pTarangs = pTarang;
	up(&Adapter->RxAppControlQueuelock);

	/* Store the Adapter structure */
	filp->private_data = pTarang;

	/*Start Queuing the control response Packets*/
	atomic_inc(&Adapter->ApplicationRunning); 
	return 0;
}
static int bcm_char_release(struct inode *inode, struct file *filp)
{
    PPER_TARANG_DATA pTarang, tmp, ptmp;
	PMINI_ADAPTER Adapter=NULL;
    struct sk_buff * pkt, * npkt;

    pTarang = (PPER_TARANG_DATA)filp->private_data;

    if(pTarang == NULL)
	{
	BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "ptarang is null\n");
	return 0;
	}

	Adapter = pTarang->Adapter;

    down( &Adapter->RxAppControlQueuelock);

    tmp = Adapter->pTarangs;
    for ( ptmp = NULL; tmp; ptmp = tmp, tmp = tmp->next )
	{
        if ( tmp == pTarang )
			break;
	}
	
    if ( tmp )
	{
        if ( !ptmp )
            Adapter->pTarangs = tmp->next;
        else
            ptmp->next = tmp->next;
	}

    else
	{
    	up( &Adapter->RxAppControlQueuelock);
	return 0;
	}

    pkt = pTarang->RxAppControlHead;
    while ( pkt )
	{
        npkt = pkt->next;
        kfree_skb(pkt);
        pkt = npkt;
	}

    up( &Adapter->RxAppControlQueuelock);

    /*Stop Queuing the control response Packets*/
    atomic_dec(&Adapter->ApplicationRunning); 

    bcm_kfree(pTarang);

	/* remove this filp from the asynchronously notified filp's */
    filp->private_data = NULL;
    return 0;
}

ssize_t bcm_char_read(struct file *filp, PCHAR buf, size_t size, loff_t *f_pos)
{
    PPER_TARANG_DATA pTarang = (PPER_TARANG_DATA)filp->private_data;
	PMINI_ADAPTER	Adapter = pTarang->Adapter;
    struct sk_buff* Packet = NULL;
    UINT            PktLen = 0;
	int 			wait_ret_val=0;

	wait_ret_val = wait_event_interruptible(Adapter->process_read_wait_queue, 
		(pTarang->RxAppControlHead || Adapter->device_removed));
	if((wait_ret_val == -ERESTARTSYS))
	{
   		BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "Exiting as i've been asked to exit!!!\n");
		return wait_ret_val;
	}

	if(Adapter->device_removed)
	{
		BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "Device Removed... Killing the Apps...\n");
		return -ENODEV;
	}
	
	if(FALSE == Adapter->fw_download_done)
		return -EACCES;

    down( &Adapter->RxAppControlQueuelock);

	if(pTarang->RxAppControlHead)
	{
		Packet = pTarang->RxAppControlHead;
		DEQUEUEPACKET(pTarang->RxAppControlHead,pTarang->RxAppControlTail);
		pTarang->AppCtrlQueueLen--;
	}

    up(&Adapter->RxAppControlQueuelock);

	if(Packet)
	{
		PktLen = Packet->len;
		if(copy_to_user(buf, Packet->data, PktLen))
		{
			bcm_kfree_skb(Packet);  
			BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "\nReturning from copy to user failure \n");
			return -EFAULT; 
		}
		BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "Read %d Bytes From Adapter packet = 0x%p by process %d!\n", PktLen, Packet, current->pid);
		bcm_kfree_skb(Packet);  
	}

    BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "<====\n");
    return PktLen;
}

static INT bcm_char_ioctl(struct inode *inode, struct file *filp, 
					UINT cmd, ULONG arg)
{
    PPER_TARANG_DATA  pTarang = (PPER_TARANG_DATA)filp->private_data;
	PMINI_ADAPTER 	Adapter = pTarang->Adapter;
	INT  			Status = STATUS_FAILURE;
	IOCTL_BUFFER 	IoBuffer={0};
#ifndef BCM_SHM_INTERFACE
    int timeout = 0;
#endif


	BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "Parameters Passed to control IOCTL cmd=0x%X arg=0x%lX", cmd, arg);

	if(_IOC_TYPE(cmd) != BCM_IOCTL)
		return -EFAULT;
	if(_IOC_DIR(cmd) & _IOC_READ)
		Status = !access_ok(VERIFY_WRITE, (PVOID)arg, _IOC_SIZE(cmd));
	else if (_IOC_DIR(cmd) & _IOC_WRITE)
	    Status = !access_ok(VERIFY_READ, (PVOID)arg, _IOC_SIZE(cmd));
	else if (_IOC_NONE == (_IOC_DIR(cmd) & _IOC_NONE))
	    Status = STATUS_SUCCESS;
	
	if(Status)
		return -EFAULT;
	
	if(Adapter->device_removed)
	{
		return -EFAULT;
	}

	if(FALSE == Adapter->fw_download_done)
	{
		switch (cmd)
		{
			case IOCTL_MAC_ADDR_REQ:
			case IOCTL_LINK_REQ:
			case IOCTL_CM_REQUEST:
			case IOCTL_SS_INFO_REQ:
			case IOCTL_SEND_CONTROL_MESSAGE:
			case IOCTL_IDLE_REQ:
				return -EACCES;
			default: 
				break;
		}
	}
	
	Status = vendorextnIoctl(Adapter, cmd, arg);
	if(Status != CONTINUE_COMMON_PATH )
	{
		 return Status;
	}

	switch(cmd){
		// Rdms for Swin Idle...
		case IOCTL_BCM_REGISTER_READ_PRIVATE:
		{
			RDM_BUFFER  sRdmBuffer = {0};	
			PCHAR temp_buff = NULL;
			UINT Bufflen = 0;
			/* Copy Ioctl Buffer structure */
			if(copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, 
				sizeof(IOCTL_BUFFER)))
			{
				Status = -EFAULT;
				break;
			}
			
			Bufflen = IoBuffer.OutputLength + (4 - IoBuffer.OutputLength%4)%4;
			temp_buff = (PCHAR)kmalloc(Bufflen, GFP_KERNEL);
			if(!temp_buff)
			{
				return STATUS_FAILURE;
			}
			if(copy_from_user(&sRdmBuffer, IoBuffer.InputBuffer, 
				IoBuffer.InputLength))
			{
				Status = -EFAULT;
				break;
			}

			down(&Adapter->rdmWrmDevAccLock); 
			Status = rdmalt(Adapter, (UINT)sRdmBuffer.Register, 
					(PUINT)temp_buff, Bufflen);
			up(&Adapter->rdmWrmDevAccLock); 
			
			if(Status != STATUS_SUCCESS)
			{
				bcm_kfree(temp_buff);
				return Status;
			}
			if(copy_to_user((PCHAR)IoBuffer.OutputBuffer, 
				(PCHAR)temp_buff, (UINT)IoBuffer.OutputLength))
			{
				Status = -EFAULT;
			}
			bcm_kfree(temp_buff);
			break;
		}
		case IOCTL_BCM_REGISTER_WRITE_PRIVATE:
		{
			WRM_BUFFER  sWrmBuffer = {0};	
			UINT uiTempVar=0;
			/* Copy Ioctl Buffer structure */
		
			if(copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, 
				sizeof(IOCTL_BUFFER)))
			{
				Status = -EFAULT;
				break;
			}
			/* Get WrmBuffer structure */
			if(copy_from_user(&sWrmBuffer, IoBuffer.InputBuffer, 
				IoBuffer.InputLength))
			{
				Status = -EFAULT;
				break;
			}
			uiTempVar = sWrmBuffer.Register & EEPROM_REJECT_MASK;
			if(!((Adapter->pstargetparams->m_u32Customize) & VSG_MODE) &&
			 	((uiTempVar == EEPROM_REJECT_REG_1)|| 
				(uiTempVar == EEPROM_REJECT_REG_2) || 
				(uiTempVar == EEPROM_REJECT_REG_3) || 
				(uiTempVar == EEPROM_REJECT_REG_4)))
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "EEPROM Access Denied, not in VSG Mode\n");
				Status = -EFAULT;
				break;
			}
			
			down(&Adapter->rdmWrmDevAccLock); 
			Status = wrmalt(Adapter, (UINT)sWrmBuffer.Register, 
						(PUINT)sWrmBuffer.Data, sizeof(B_UINT32));
			up(&Adapter->rdmWrmDevAccLock); 

			if(Status == STATUS_SUCCESS)
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"WRM Done\n");
			}
			else
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "WRM Failed\n");
				Status = -EFAULT;
			}
			break;
		}

		case IOCTL_BCM_REGISTER_READ:
		case IOCTL_BCM_EEPROM_REGISTER_READ:
		{
			RDM_BUFFER  sRdmBuffer = {0};	
			PCHAR temp_buff = NULL;
			UINT uiTempVar = 0;
			
			if((Adapter->IdleMode == TRUE) ||
				(Adapter->bShutStatus ==TRUE) ||
				(Adapter->bPreparingForLowPowerMode ==TRUE))
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Device in Idle Mode, Blocking Rdms\n");
				Status = -EACCES;
				break;
			}

			/* Copy Ioctl Buffer structure */
			if(copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, 
				sizeof(IOCTL_BUFFER)))
			{
				Status = -EFAULT;
				break;
			}

			temp_buff = (PCHAR)kmalloc(IoBuffer.OutputLength, GFP_KERNEL);
			if(!temp_buff)
			{
				return STATUS_FAILURE;
			}
			if(copy_from_user(&sRdmBuffer, IoBuffer.InputBuffer, 
				IoBuffer.InputLength))
			{
				Status = -EFAULT;
				break;
			}
#ifdef BCM_SHM_INTERFACE
			sRdmBuffer.Register |= 0xA0000000;
#endif
			if(
#if !defined(BCM_SHM_INTERFACE)				
				(((B_UINT32)sRdmBuffer.Register & 0x0F000000) != 0x0F000000) || 
#else
									(
				((B_UINT32)sRdmBuffer.Register  < 0xAF000000) 
			   							||
			    (((B_UINT32)sRdmBuffer.Register > 0xBFFFFFFF))
									)||
#endif				
					((B_UINT32)sRdmBuffer.Register & 0x3)
			  )
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "RDM Done On invalid Address : %x Access Denied.\n",
					(int)sRdmBuffer.Register);
				Status = -EINVAL;
				break;
			}

			uiTempVar = sRdmBuffer.Register & EEPROM_REJECT_MASK;

			down(&Adapter->rdmWrmDevAccLock); 
			Status = rdmaltWithLock(Adapter, (UINT)sRdmBuffer.Register, 
						(PUINT)temp_buff, IoBuffer.OutputLength);
			up(&Adapter->rdmWrmDevAccLock);
			
			if(Status != STATUS_SUCCESS)
			{
				bcm_kfree(temp_buff);
				return Status;
			}
			if(copy_to_user((PCHAR)IoBuffer.OutputBuffer, 
				(PCHAR)temp_buff, (UINT)IoBuffer.OutputLength))
			{
				Status = -EFAULT;
			}
			bcm_kfree(temp_buff);
			break;
		}
		case IOCTL_BCM_REGISTER_WRITE:
		case IOCTL_BCM_EEPROM_REGISTER_WRITE:
		{
			WRM_BUFFER  sWrmBuffer = {0};	
			UINT uiTempVar=0;

			if((Adapter->IdleMode == TRUE) ||
				(Adapter->bShutStatus ==TRUE) ||
				(Adapter->bPreparingForLowPowerMode ==TRUE))
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Device in Idle Mode, Blocking Wrms\n");
				Status = -EACCES;
				break;
			}
			/* Copy Ioctl Buffer structure */
			if(copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, 
					sizeof(IOCTL_BUFFER)))
			{
				Status = -EFAULT;
				break;
			}
			/* Get WrmBuffer structure */
			if(copy_from_user(&sWrmBuffer, IoBuffer.InputBuffer, 
				IoBuffer.InputLength))
			{
				Status = -EFAULT;
				break;
			}

#ifdef BCM_SHM_INTERFACE
			sWrmBuffer.Register |= 0xA0000000;
#endif
			if(
#if !defined(BCM_SHM_INTERFACE)	

				(((B_UINT32)sWrmBuffer.Register & 0x0F000000) != 0x0F000000) || 
#else
									(
				((B_UINT32)sWrmBuffer.Register  < 0xAF000000) 
			   							||
			    (((B_UINT32)sWrmBuffer.Register > 0xBFFFFFFF))
									)||
#endif				
					((B_UINT32)sWrmBuffer.Register & 0x3)
			 )
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "WRM Done On invalid Address : %x Access Denied.\n",
						(int)sWrmBuffer.Register);
				Status = -EINVAL;
				break;
			}
			uiTempVar = sWrmBuffer.Register & EEPROM_REJECT_MASK;
			if(!((Adapter->pstargetparams->m_u32Customize) & VSG_MODE) &&
				((uiTempVar == EEPROM_REJECT_REG_1)|| 
				(uiTempVar == EEPROM_REJECT_REG_2) || 
				(uiTempVar == EEPROM_REJECT_REG_3) || 
				(uiTempVar == EEPROM_REJECT_REG_4)) &&
				(cmd == IOCTL_BCM_REGISTER_WRITE))
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "EEPROM Access Denied, not in VSG Mode\n");
				Status = -EFAULT;
				break;
			}

			down(&Adapter->rdmWrmDevAccLock); 
			Status = wrmaltWithLock(Adapter, (UINT)sWrmBuffer.Register, 
							(PUINT)sWrmBuffer.Data, sWrmBuffer.Length);
			up(&Adapter->rdmWrmDevAccLock);

			if(Status == STATUS_SUCCESS)
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, OSAL_DBG, DBG_LVL_ALL, "WRM Done\n");
			}
			else
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "WRM Failed\n");
				Status = -EFAULT;
			}
			break;
		}
		case IOCTL_BCM_GPIO_SET_REQUEST:
		{
			UCHAR ucResetValue[4];
			UINT value =0; 
			UINT uiBit = 0;
	        UINT uiOperation = 0;
            BOOLEAN bgpios = WIMAX_IDX; //working on GPIO_WIMAX
			
			GPIO_INFO   gpio_info = {0};
			if((Adapter->IdleMode == TRUE) || 
				(Adapter->bShutStatus ==TRUE) || 
				(Adapter->bPreparingForLowPowerMode ==TRUE))
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"GPIO Can't be set/clear in Low power Mode");
				Status = -EACCES;
				break;
			}
			if(copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, sizeof(IOCTL_BUFFER)))
			{
				Status = -EFAULT;
				break;
		    }	
			if(copy_from_user(&gpio_info, IoBuffer.InputBuffer, IoBuffer.InputLength))
			{
				Status = -EFAULT;
				break;
			}
			uiBit  = gpio_info.uiGpioNumber;
			uiOperation = gpio_info.uiGpioValue;

			value= (1<<uiBit);
			
			if(IsReqGpioIsLedInNVM(Adapter,value,bgpios) ==FALSE)
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "Sorry, Requested GPIO<0x%X> is not correspond to LED !!!",value);
				Status = -EINVAL;
				break;
			}
				
			down(&Adapter->fw_download_sema);/*take this lock to disable GPIO registrers before we get the start interrupt 
												after firmware download completes -- take care of the scenario when 
												the user issues the IOCTL to firware download, since this semaphore is 
												valid only in that case.*/
			do{
				if(uiOperation)//Set - setting 1
				{					
					//Set the gpio output register
					Status = wrmaltWithLock(Adapter,BCM_GPIO_OUTPUT_SET_REG , 
							(PUINT)(&value), sizeof(UINT));
					if(Status == STATUS_SUCCESS)
					{
	               	    BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "Set the GPIO bit\n");
					}
	        	    else
			        {
	                   	BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"Failed to set the %dth GPIO \n",uiBit);
	                   	break;
	               	}
				}
				else//Unset - setting 0
				{
					//Set the gpio output register
					Status = wrmaltWithLock(Adapter,BCM_GPIO_OUTPUT_CLR_REG , 
							(PUINT)(&value), sizeof(UINT));
					if(Status == STATUS_SUCCESS)
					{
	               	    BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"Set the GPIO bit\n");
					}
	        	    else
			        {
	                   	BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"Failed to clear the %dth GPIO \n",uiBit);
	                   	break;
	               	}
				}

				Status = rdmaltWithLock(Adapter, (UINT)GPIO_MODE_REGISTER, 
						(PUINT)ucResetValue, sizeof(UINT));
				if (STATUS_SUCCESS != Status)
	            {
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"GPIO_MODE_REGISTER read failed");
					break;
				}
				//Set the gpio mode register to output 
				*(UINT*)ucResetValue |= (1<<uiBit);
				Status = wrmaltWithLock(Adapter,GPIO_MODE_REGISTER , 
						(PUINT)ucResetValue, sizeof(UINT));
				if(Status == STATUS_SUCCESS)
				{
	            	BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"Set the GPIO to output Mode\n");
				}
	            else
	            {
	            	BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "Failed to put GPIO in Output Mode\n");
	                break;
	            } 
			}while(0);
			up(&Adapter->fw_download_sema);
		}
		break;
		case BCM_LED_THREAD_STATE_CHANGE_REQ:
		{

			USER_THREAD_REQ threadReq = {0};
			BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"User made LED thread InActive");
			
			if((Adapter->IdleMode == TRUE) || 
				(Adapter->bShutStatus ==TRUE) || 
				(Adapter->bPreparingForLowPowerMode ==TRUE))
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"GPIO Can't be set/clear in Low power Mode");
				Status = -EACCES;
				break;
			}
			Status =copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, sizeof(IOCTL_BUFFER));
			if(Status)
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Failed while copying the IOBufer from user space err:%d",Status);
				break;
			}	

			Status= copy_from_user(&threadReq, IoBuffer.InputBuffer, IoBuffer.InputLength);
			if(Status)
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Failed while copying the InputBuffer from user space err:%d",Status);
				break;
			}
			//if LED thread is running(Actively or Inactively) set it state to make inactive
			if(Adapter->LEDInfo.led_thread_running)
			{
				if(threadReq.ThreadState == LED_THREAD_ACTIVATION_REQ)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"Activating thread req");
					Adapter->DriverState = LED_THREAD_ACTIVE;
				}	
				else
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"DeActivating Thread req.....");
					Adapter->DriverState = LED_THREAD_INACTIVE;
				}	

				//signal thread.
				wake_up(&Adapter->LEDInfo.notify_led_event);
				
			}
		}
		break;
		case IOCTL_BCM_GPIO_STATUS_REQUEST:
		{
			B_UINT32 uiBit = 0;
			UCHAR ucRead[4];
			GPIO_INFO   gpio_info = {0};
			if((Adapter->IdleMode == TRUE) ||
				(Adapter->bShutStatus ==TRUE) ||
				(Adapter->bPreparingForLowPowerMode ==TRUE))
			{
				Status = -EACCES;
				break;
			}
			if(copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, sizeof(IOCTL_BUFFER)))
            {
            	Status = -EFAULT;
                    break;
                }
                if(copy_from_user(&gpio_info, IoBuffer.InputBuffer, IoBuffer.InputLength))
                {
                    Status = -EFAULT;
                    break;
                }
                uiBit  = gpio_info.uiGpioNumber;
				  //Set the gpio output register
				Status = rdmaltWithLock(Adapter, (UINT)GPIO_PIN_STATE_REGISTER,
                	(PUINT)ucRead, sizeof(UINT));
                if(Status != STATUS_SUCCESS)
                {
                    BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "RDM Failed\n");
					return Status;
                }

			}
			break;
			case IOCTL_BCM_GPIO_MULTI_REQUEST:
          	{

                //using same structure for both GPIO and SPIO

                UCHAR ucResetValue[4];
				GPIO_MULTI_INFO gpio_multi_info[MAX_IDX];
				PGPIO_MULTI_INFO pgpio_multi_info = (PGPIO_MULTI_INFO)gpio_multi_info;
                
                UINT RegArry[MAX_IDX][TOTAL_REG_IN_MULTI_REQUEST]=

                    {
                      {
                        (UINT)BCM_GPIO_OUTPUT_SET_REG,
                        (UINT)BCM_GPIO_OUTPUT_CLR_REG,
                        (UINT)GPIO_PIN_STATE_REGISTER,
                      },

                      {0},

                      {
                        (UINT)BCM_SPIO_OUTPUT_SET_REG,
                        (UINT)BCM_SPIO_OUTPUT_CLR_REG,
                        (UINT)SPIO_PIN_STATE_REGISTER,
                      }
                        
                    };

                int totalIndex;
                
                memset( pgpio_multi_info, 0, MAX_IDX * sizeof(GPIO_MULTI_INFO));
		
				if((Adapter->IdleMode == TRUE) ||
				(Adapter->bShutStatus ==TRUE) ||
				(Adapter->bPreparingForLowPowerMode ==TRUE))
				{
					Status = -EINVAL;
					break;
				}
				Status = copy_from_user( (PCHAR)&IoBuffer, ( PCHAR)arg, sizeof( IOCTL_BUFFER));
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Failed while copying the IOBufer from user space err:%d",Status);
					break;
				}

				Status = copy_from_user( &gpio_multi_info, IoBuffer.InputBuffer, IoBuffer.InputLength);
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Failed while copying the IOBufer Contents from user space err:%d",Status);
					break;
				}
                
                //Validating the request 

                if(!IsReqGpioIsLedInNVM(Adapter,pgpio_multi_info[WIMAX_IDX].uiGPIOMask,WIMAX_IDX)||
                   !IsReqGpioIsLedInNVM(Adapter,pgpio_multi_info[WIMAX_SPIO_IDX].uiGPIOMask,WIMAX_SPIO_IDX))
    			{
	    		   
	    			BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, LED_DUMP_INFO, DBG_LVL_ALL, "Sorry, Requested GPIO<0x%X> is not correspond to NVM LED bit map<0x%X>!!!",pgpio_multi_info[WIMAX_IDX].uiGPIOMask,Adapter->gpioBitMap);
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, LED_DUMP_INFO, DBG_LVL_ALL, "Sorry, Requested SPIO<0x%X> is not correspond to NVM LED bit map<0x%X>!!!",pgpio_multi_info[WIMAX_SPIO_IDX].uiGPIOMask,Adapter->spioBitMap);
		
		    		Status = -EINVAL;
                    return Status;
                }

                
                totalIndex = IoBuffer.InputLength/sizeof(GPIO_MULTI_INFO);
 
                totalIndex --;


                while (totalIndex >= 0)
                {

                    if(totalIndex == HOST_IDX)
                    {
                        totalIndex--;
                        continue;
                    }

                    BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, LED_DUMP_INFO, DBG_LVL_ALL, "Requested Command:<0x%X> !!!\n",pgpio_multi_info[totalIndex].uiGPIOCommand);
                    BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, LED_DUMP_INFO, DBG_LVL_ALL, "Requested Mask   :<0x%X> !!!\n",pgpio_multi_info[totalIndex].uiGPIOMask);                    
                    BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, LED_DUMP_INFO, DBG_LVL_ALL, "Requested Value  :<0x%X> !!!\n",pgpio_multi_info[totalIndex].uiGPIOValue);                    
                    
            			                                
    				/* Set the gpio output register */
		
	    			if( ( pgpio_multi_info[totalIndex].uiGPIOMask) &
		    			( pgpio_multi_info[totalIndex].uiGPIOCommand))
			    	{
        				/* Set 1's in GPIO OUTPUT REGISTER */
        				*(UINT*) ucResetValue =  pgpio_multi_info[totalIndex].uiGPIOMask &
        				        				 pgpio_multi_info[totalIndex].uiGPIOCommand &
        										 pgpio_multi_info[totalIndex].uiGPIOValue;
        	
        				if( *(UINT*) ucResetValue)
        					Status = wrmaltWithLock( Adapter,RegArry[totalIndex][OUTPUT_SET_REG_INDEX], (PUINT) ucResetValue, sizeof(B_UINT32));
        	
        				if( Status != STATUS_SUCCESS)
        				{
        					BCM_DEBUG_PRINT( Adapter,DBG_TYPE_PRINTK, 0, 0,"WRM to BCM_GPIO_OUTPUT_SET_REG Failed.");
        					return Status;
        				}
        	
        				/* Clear to 0's in GPIO OUTPUT REGISTER */
        				*(UINT*) ucResetValue = (pgpio_multi_info[totalIndex].uiGPIOMask &
        						pgpio_multi_info[totalIndex].uiGPIOCommand &
        						( ~( pgpio_multi_info[totalIndex].uiGPIOValue)));
        	
        				if( *(UINT*) ucResetValue)
        					Status = wrmaltWithLock( Adapter,RegArry[totalIndex][OUTPUT_CLR_REG_INDEX], (PUINT) ucResetValue, sizeof(B_UINT32));
        	
        				if( Status != STATUS_SUCCESS)
        				{
        					BCM_DEBUG_PRINT( Adapter,DBG_TYPE_PRINTK, 0, 0,"WRM to BCM_GPIO_OUTPUT_CLR_REG Failed." );
        					return Status;
        				}
				    }
		
    				if( pgpio_multi_info[totalIndex].uiGPIOMask)
    				{
    					Status = rdmaltWithLock(Adapter,RegArry[totalIndex][PIN_STATE_REGISTER_INDEX], (PUINT)ucResetValue, sizeof(UINT));
    		
    					if(Status != STATUS_SUCCESS)
    					{
    						BCM_DEBUG_PRINT( Adapter,DBG_TYPE_PRINTK, 0, 0,"RDM to GPIO_PIN_STATE_REGISTER Failed.");
    						return Status;
    					}
    		
    					pgpio_multi_info[totalIndex].uiGPIOValue = ( *(UINT*)ucResetValue & 
    											pgpio_multi_info[totalIndex].uiGPIOMask);
    				}

                    totalIndex--;
                }
    		
				Status = copy_to_user( (PCHAR)IoBuffer.OutputBuffer, &gpio_multi_info, IoBuffer.OutputLength);
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Failed while copying Content to IOBufer for user space err:%d",Status);
					break;
				}
			}
			break;
		case IOCTL_BCM_GPIO_MODE_REQUEST:
        {
		     //using same structure for both GPIO and SPIO
            
			UCHAR ucResetValue[4];
			GPIO_MULTI_MODE gpio_multi_mode[MAX_IDX];
			PGPIO_MULTI_MODE pgpio_multi_mode = ( PGPIO_MULTI_MODE) gpio_multi_mode;
            UINT RegArry[MAX_IDX][TOTAL_REG_IN_MODE_REQUEST]=
                {
                    {
                        (UINT)GPIO_MODE_REGISTER,
                    },
                    
                    {0},

                    {
                        (UINT)SPIO_MODE_REGISTER,
                    }                                                                        
                };

            
            int totalIndex;

            memset(pgpio_multi_mode, 0, MAX_IDX * sizeof(GPIO_MULTI_MODE));

			if((Adapter->IdleMode == TRUE) ||
				(Adapter->bShutStatus ==TRUE) ||
				(Adapter->bPreparingForLowPowerMode ==TRUE))
			{
				Status = -EINVAL;
				break;
			}
			Status = copy_from_user( (PCHAR)&IoBuffer, ( PCHAR)arg, sizeof( IOCTL_BUFFER));
			if(Status)
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Failed while copying the IOBufer from user space err:%d",Status);
				break;
			}

			Status = copy_from_user( &gpio_multi_mode, IoBuffer.InputBuffer, IoBuffer.InputLength);
			if(Status)
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Failed while copying the IOBufer Contents from user space err:%d",Status);
				break;
			}

            //Validating the request            

            if(!IsReqGpioIsLedInNVM(Adapter,pgpio_multi_mode[WIMAX_IDX].uiGPIOMask, WIMAX_IDX) || 
               !IsReqGpioIsLedInNVM(Adapter,pgpio_multi_mode[WIMAX_SPIO_IDX].uiGPIOMask, WIMAX_SPIO_IDX))
			{
   				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, LED_DUMP_INFO, DBG_LVL_ALL, "Sorry, Requested GPIO<0x%X> is not correspond to NVM LED bit map<0x%X>!!!",pgpio_multi_mode[WIMAX_IDX].uiGPIOMask,Adapter->gpioBitMap);
	    		BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, LED_DUMP_INFO, DBG_LVL_ALL, "Sorry, Requested SPIO<0x%X> is not correspond to NVM LED bit map<0x%X>!!!",pgpio_multi_mode[WIMAX_SPIO_IDX].uiGPIOMask,Adapter->spioBitMap);
				Status = -EINVAL;
				break;
			}

            
            totalIndex = IoBuffer.InputLength/sizeof(GPIO_MULTI_MODE);

            totalIndex --;
            
            while (totalIndex >= 0)
            {

                if(totalIndex == HOST_IDX)
                {
                    totalIndex--;
                    continue;
                }
				
                down(&Adapter->fw_download_sema);/*take this lock to disable GPIO registrers before 
                											we get the start interrupt after firmware download 
													completes -- take care of the scenario when the user 
													issues the IOCTL to firware download,
													since this semaphore is valid only in that case*/
    			Status = rdmaltWithLock( Adapter,RegArry[totalIndex][MODE_REGISTER_INDEX], ( PUINT) ucResetValue, sizeof( UINT));
                
    			if( STATUS_SUCCESS != Status)
    			{
    				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Read of GPIO_MODE_REGISTER failed");
					up(&Adapter->fw_download_sema);
    				return Status;
    			}

                BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, LED_DUMP_INFO, DBG_LVL_ALL, "Requested Mode:<0x%X> !!!\n",pgpio_multi_mode[totalIndex].uiGPIOMask);
                BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, LED_DUMP_INFO, DBG_LVL_ALL, "Requested Mask:<0x%X> !!!\n",pgpio_multi_mode[totalIndex].uiGPIOMask);
                
    		   		
    			if( pgpio_multi_mode[totalIndex].uiGPIOMask)
    			{
    				/* write all OUT's (1's) */
    				*( UINT*) ucResetValue |= ( pgpio_multi_mode[totalIndex].uiGPIOMode & 
    								pgpio_multi_mode[totalIndex].uiGPIOMask);
    				/* write all IN's (0's) */
    				*( UINT*) ucResetValue &= ~( ( ~pgpio_multi_mode[totalIndex].uiGPIOMode) & 
    								pgpio_multi_mode[totalIndex].uiGPIOMask);
    		 
    				/* Currently implemented return the modes of all GPIO's 
    				 * else needs to bit AND with  mask
    				 * */
    				pgpio_multi_mode[totalIndex].uiGPIOMode = *(UINT*)ucResetValue;
    	
    				Status = wrmaltWithLock( Adapter,RegArry[totalIndex][MODE_REGISTER_INDEX], ( PUINT) ucResetValue, sizeof( B_UINT32));

					if( Status == STATUS_SUCCESS)
    				{
    					BCM_DEBUG_PRINT( Adapter,DBG_TYPE_OTHERS, LED_DUMP_INFO, DBG_LVL_ALL, "WRM to GPIO_MODE_REGISTER Done");
    				}
    				else
    				{
    					BCM_DEBUG_PRINT( Adapter,DBG_TYPE_PRINTK, 0, 0,"WRM to GPIO_MODE_REGISTER Failed");
						up(&Adapter->fw_download_sema);
    					Status = -EFAULT;
    					break;
    				}
    			}
    			else /* if uiGPIOMask is 0 then return mode register configuration */
    			{
					pgpio_multi_mode[totalIndex].uiGPIOMode = *( UINT*) ucResetValue;
    			}
				
				up(&Adapter->fw_download_sema);
                totalIndex--;
            }

			if(Status != STATUS_SUCCESS)
				break;
			
			Status = copy_to_user( (PCHAR)IoBuffer.OutputBuffer, &gpio_multi_mode, IoBuffer.OutputLength);
			if(Status)
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Failed while copying Content to IOBufer for user space err:%d",Status);
				break;
			}
		}
		break;
		
		case IOCTL_MAC_ADDR_REQ:
		case IOCTL_LINK_REQ:
		case IOCTL_CM_REQUEST:
		case IOCTL_SS_INFO_REQ:
		case IOCTL_SEND_CONTROL_MESSAGE:
		case IOCTL_IDLE_REQ:
		{
			PVOID pvBuffer=NULL;
			/* Copy Ioctl Buffer structure */
			if(copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, 
							sizeof(IOCTL_BUFFER)))
			{
				Status = -EFAULT;
				break;
			}
			pvBuffer=kmalloc(IoBuffer.InputLength, GFP_KERNEL);
			if(!pvBuffer)
			{
				return -ENOMEM;
			}
			
			if(copy_from_user(pvBuffer, IoBuffer.InputBuffer, 
					IoBuffer.InputLength))
			{
				Status = -EFAULT;
				bcm_kfree(pvBuffer);
				break;
			}
			
			down(&Adapter->LowPowerModeSync);
			Status = wait_event_interruptible_timeout(Adapter->lowpower_mode_wait_queue,
													!Adapter->bPreparingForLowPowerMode, 
													(1 * HZ));
			if(Status == -ERESTARTSYS)
					goto cntrlEnd;
			
			if(Adapter->bPreparingForLowPowerMode)
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "Preparing Idle Mode is still True - Hence Rejecting control message\n");
				Status = STATUS_FAILURE ;
				goto cntrlEnd ;
			}
			Status = CopyBufferToControlPacket(Adapter, (PVOID)pvBuffer);
		cntrlEnd:	
			up(&Adapter->LowPowerModeSync);
			bcm_kfree(pvBuffer);
			break;
		}
#ifndef BCM_SHM_INTERFACE
		case IOCTL_BCM_BUFFER_DOWNLOAD_START:
		{
			INT NVMAccess = down_trylock(&Adapter->NVMRdmWrmLock) ;
			if(NVMAccess)
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, " IOCTL_BCM_CHIP_RESET not allowed as EEPROM Read/Write is in progress\n");	
				return -EACCES;
			}
			BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "Starting the firmware download PID =0x%x!!!!\n", current->pid);
		    if(!down_trylock(&Adapter->fw_download_sema))
			{
				Adapter->bBinDownloaded=FALSE;
				Adapter->fw_download_process_pid=current->pid;
				Adapter->bCfgDownloaded=FALSE;
				Adapter->fw_download_done=FALSE;
				netif_carrier_off(Adapter->dev);			
				netif_stop_queue(Adapter->dev);
				down(&Adapter->rdmWrmDevAccLock);  
				Status = reset_card_proc(Adapter);

				//Initializing the NVM. 
				BcmInitNVM(Adapter);
				up(&Adapter->rdmWrmDevAccLock); 
				
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "reset_card_proc Failed!\n");
					up(&Adapter->fw_download_sema);
					up(&Adapter->NVMRdmWrmLock);
					break;
				}
				mdelay(10);
			}
			else
			{

				Status = -EBUSY;
				
			}
			up(&Adapter->NVMRdmWrmLock);
			break;
		}
		case IOCTL_BCM_BUFFER_DOWNLOAD:
			{
				FIRMWARE_INFO 	*psFwInfo=NULL;
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "Starting the firmware download PID =0x%x!!!!\n", current->pid);
			do{
				if(!down_trylock(&Adapter->fw_download_sema))
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Invalid way to download buffer. Use Start and then call this!!!\n");
					Status=-EINVAL;
					break;
				}
				/* Copy Ioctl Buffer structure */
				if(copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, 
						sizeof(IOCTL_BUFFER)))
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "copy_from_user 1 failed\n");
					Status = -EFAULT;
					break;
				}
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "Length for FW DLD is : %x\n", 
										IoBuffer.InputLength);
				psFwInfo=kmalloc(sizeof(*psFwInfo), GFP_KERNEL);
				if(!psFwInfo)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Failed to allocate buffer!!!!\n");
					Status = -ENOMEM;
					break;
				}
				if(copy_from_user(psFwInfo, IoBuffer.InputBuffer, 
							IoBuffer.InputLength))
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Copy_from_user 2 failed\n");
					Status = -EFAULT;
					break;
				}
				
				if(!psFwInfo->pvMappedFirmwareAddress || 
						(psFwInfo->u32FirmwareLength == 0))
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Something else is wrong %u\n", 
					psFwInfo->u32FirmwareLength);
					Status = -EINVAL;
					break;
				}
				

				down(&Adapter->rdmWrmDevAccLock); 
				Status = bcm_ioctl_fw_download(Adapter, psFwInfo);
				up(&Adapter->rdmWrmDevAccLock); 
				
				if(Status != STATUS_SUCCESS)
				{
					if(psFwInfo->u32StartingAddress==CONFIG_BEGIN_ADDR)
					{
						BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "IOCTL: Configuration File Upload Failed\n");
					}
					else
					{
						BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "IOCTL: Firmware File Upload Failed\n");
					}
					//up(&Adapter->fw_download_sema);

					if(Adapter->LEDInfo.led_thread_running & BCM_LED_THREAD_RUNNING_ACTIVELY)
					{
						Adapter->DriverState = DRIVER_INIT;
						Adapter->LEDInfo.bLedInitDone = FALSE;
						wake_up(&Adapter->LEDInfo.notify_led_event);
					}
				}
				break ;
			  }while(0);	

			  if(Status != STATUS_SUCCESS)	
					up(&Adapter->fw_download_sema);
			  
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "IOCTL: Firmware File Uploaded\n");
				bcm_kfree(psFwInfo);
				break;
			}
		case IOCTL_BCM_BUFFER_DOWNLOAD_STOP:
		{	
			INT NVMAccess = down_trylock(&Adapter->NVMRdmWrmLock);
			if(NVMAccess)
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, " FW download blocked as EEPROM Read/Write is in progress\n");	
				up(&Adapter->fw_download_sema);
				return -EACCES;
			}
			if(down_trylock(&Adapter->fw_download_sema))
			{
				Adapter->bBinDownloaded=TRUE;
				Adapter->bCfgDownloaded=TRUE;
				atomic_set(&Adapter->CurrNumFreeTxDesc, 0);
				atomic_set(&Adapter->RxRollOverCount, 0);
				Adapter->CurrNumRecvDescs=0;
				Adapter->uiFirstInterrupt = 0;

				//setting the Mips to Run
				
				down(&Adapter->rdmWrmDevAccLock); 
				Status = run_card_proc(Adapter);
				up(&Adapter->rdmWrmDevAccLock); 
				
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Firm Download Failed\n");
					up(&Adapter->fw_download_sema);
					up(&Adapter->NVMRdmWrmLock);
					break;
				}
				else
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "Firm Download Over...\n");
				mdelay(10);
				/* Wait for MailBox Interrupt */
				if(StartInterruptUrb((PS_INTERFACE_ADAPTER)Adapter->pvInterfaceAdapter))
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Unable to send interrupt...\n");
				}
				timeout = 5*HZ;
				Adapter->waiting_to_fw_download_done = FALSE;
				wait_event_timeout(Adapter->ioctl_fw_dnld_wait_queue, 
					Adapter->waiting_to_fw_download_done, timeout);
				Adapter->fw_download_process_pid=INVALID_PID;
				Adapter->fw_download_done=TRUE;
				atomic_set(&Adapter->CurrNumFreeTxDesc, 0);
				Adapter->CurrNumRecvDescs = 0;
				Adapter->PrevNumRecvDescs = 0;
				atomic_set(&Adapter->cntrlpktCnt,0);
                Adapter->LinkUpStatus = 0;
                Adapter->LinkStatus = 0;
				
				if(Adapter->LEDInfo.led_thread_running & BCM_LED_THREAD_RUNNING_ACTIVELY)
				{
					Adapter->DriverState = FW_DOWNLOAD_DONE;
					wake_up(&Adapter->LEDInfo.notify_led_event);
				}
				
				if(!timeout)
				{
					Status = -ENODEV;
				}
			}
			else
			{
			   	Status = -EINVAL;
			}
			up(&Adapter->fw_download_sema);
			up(&Adapter->NVMRdmWrmLock);
			break;
		}
#endif    
		case IOCTL_BE_BUCKET_SIZE:
			Adapter->BEBucketSize = *(PULONG)arg;
			Status = STATUS_SUCCESS;
			break;

		case IOCTL_RTPS_BUCKET_SIZE:
			Adapter->rtPSBucketSize = *(PULONG)arg;
			Status = STATUS_SUCCESS;
			break;
		case IOCTL_CHIP_RESET:
	    {
			INT NVMAccess = down_trylock(&Adapter->NVMRdmWrmLock);
			if(NVMAccess)
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, " IOCTL_BCM_CHIP_RESET not allowed as EEPROM Read/Write is in progress\n");	
				return -EACCES;
			}

			
			down(&Adapter->RxAppControlQueuelock);

			down(&Adapter->rdmWrmDevAccLock); 
			Status = reset_card_proc(Adapter);

			//Initializing the NVM. 
			BcmInitNVM(Adapter);
			up(&Adapter->rdmWrmDevAccLock);
			
			flushAllAppQ();
			up(&Adapter->RxAppControlQueuelock);
			up(&Adapter->NVMRdmWrmLock);
			ResetCounters(Adapter);
			break;
		}
		case IOCTL_QOS_THRESHOLD:
		{
			USHORT uiLoopIndex;
			for(uiLoopIndex = 0 ; uiLoopIndex < NO_OF_QUEUES ; uiLoopIndex++)
			{
				Adapter->PackInfo[uiLoopIndex].uiThreshold = *(PULONG)arg;
			}
			Status = STATUS_SUCCESS;
			break;
		}

		case IOCTL_DUMP_PACKET_INFO:
			
			DumpPackInfo(Adapter);
         	DumpPhsRules(&Adapter->stBCMPhsContext);
			Status = STATUS_SUCCESS;
			break;

		case IOCTL_GET_PACK_INFO:
			if(copy_to_user((PCHAR)arg, &Adapter->PackInfo, 
				sizeof(PacketInfo)*NO_OF_QUEUES))
			{
				Status = -EFAULT;
				break;
			}
			Status = STATUS_SUCCESS;  
			break;
		case IOCTL_BCM_SWITCH_TRANSFER_MODE:
		{		
			UINT uiData = 0;
			if(copy_from_user(&uiData, (PUINT)arg, sizeof(UINT)))
			{
				Status = -EFAULT;
				break;
			}
			if(uiData)	/* Allow All Packets */
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "IOCTL_BCM_SWITCH_TRANSFER_MODE: ETH_PACKET_TUNNELING_MODE\n");
				Adapter->TransferMode = ETH_PACKET_TUNNELING_MODE;
			}
			else	/* Allow IP only Packets */
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "IOCTL_BCM_SWITCH_TRANSFER_MODE: IP_PACKET_ONLY_MODE\n");
				Adapter->TransferMode = IP_PACKET_ONLY_MODE;
			}
			Status = STATUS_SUCCESS;
			break;
		}

		case IOCTL_BCM_GET_DRIVER_VERSION:
		{
			/* Copy Ioctl Buffer structure */
			if(copy_from_user((PCHAR)&IoBuffer, 
					(PCHAR)arg, sizeof(IOCTL_BUFFER)))
			{
				Status = -EFAULT;
				break;
			}
			if(copy_to_user((PUCHAR)IoBuffer.OutputBuffer, 
				VER_FILEVERSION_STR, (UINT)IoBuffer.OutputLength))
			{
				Status = -EFAULT;
				break;
			}
			Status = STATUS_SUCCESS; 
			break;
		}		
		case IOCTL_BCM_GET_CURRENT_STATUS:
		{
			LINK_STATE *plink_state = NULL;
			/* Copy Ioctl Buffer structure */
			if(copy_from_user((PCHAR)&IoBuffer, 
					(PCHAR)arg, sizeof(IOCTL_BUFFER)))
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "copy_from_user failed..\n");
				Status = -EFAULT;
				break;
			}
			plink_state = (LINK_STATE*)arg;	
			plink_state->bIdleMode = (UCHAR)Adapter->IdleMode;
			plink_state->bShutdownMode = Adapter->bShutStatus;
			plink_state->ucLinkStatus = (UCHAR)Adapter->LinkStatus;
			if(copy_to_user((PUCHAR)IoBuffer.OutputBuffer, 
				(PUCHAR)plink_state, (UINT)IoBuffer.OutputLength))
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Copy_to_user Failed..\n");
				Status = -EFAULT;
				break;
			}
			Status = STATUS_SUCCESS; 
			break;
		}
        case IOCTL_BCM_SET_MAC_TRACING:
        {
            UINT  tracing_flag;
            /* copy ioctl Buffer structure */
			if(copy_from_user((PCHAR)&IoBuffer, 
				(PCHAR)arg, sizeof(IOCTL_BUFFER)))
			{
				Status = -EFAULT;
				break;
			}
			if(copy_from_user((PCHAR)&tracing_flag,
                     (PCHAR)IoBuffer.InputBuffer,sizeof(UINT))) 
            {
				Status = -EFAULT;
				break;
			}
            if (tracing_flag)
                Adapter->pTarangs->MacTracingEnabled = TRUE;
            else
                Adapter->pTarangs->MacTracingEnabled = FALSE;
            break;
        }
		case IOCTL_BCM_GET_DSX_INDICATION:
		{
			B_UINT32 ulSFId=0;
			if(copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, 
					sizeof(IOCTL_BUFFER)))
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Invalid IO buffer!!!" );
				Status = -EFAULT;
				break;
			}
			if(IoBuffer.OutputLength < sizeof(stLocalSFAddIndicationAlt))
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Mismatch req: %x needed is =0x%lx!!!", 
					IoBuffer.OutputLength,(ULONG) (sizeof(stLocalSFAddIndicationAlt)));
				return -EINVAL;
			}
			if(copy_from_user((PCHAR)&ulSFId, (PCHAR)IoBuffer.InputBuffer, 
					sizeof(ulSFId)))
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Invalid SFID!!! %u", ulSFId );
				Status = -EFAULT;
				break;
			}
			BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "Get DSX Data SF ID is =%x\n", ulSFId );
			get_dsx_sf_data_to_application(Adapter, ulSFId, 
				IoBuffer.OutputBuffer);
			Status=STATUS_SUCCESS;
		}
		break;
		case IOCTL_BCM_GET_HOST_MIBS:
		{
			PCHAR temp_buff;

			if(copy_from_user((PCHAR)&IoBuffer, 
					(PCHAR)arg, sizeof(IOCTL_BUFFER)))
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Copy_from user for IoBuff failed\n");
				Status = -EFAULT;
				break;
			}

			if(IoBuffer.OutputLength != sizeof(S_MIBS_HOST_STATS_MIBS))
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Length Check failed %u %ld\n", IoBuffer.OutputLength, 
										(ULONG)(sizeof(S_MIBS_HOST_STATS_MIBS)));
	          	return -EINVAL;
			}
			
			temp_buff = (PCHAR)kmalloc(IoBuffer.OutputLength, GFP_KERNEL);
	
			if(!temp_buff)
			{
				return STATUS_FAILURE;
			}

			Status = ProcessGetHostMibs(Adapter, 
					(PUCHAR)temp_buff, IoBuffer.OutputLength);

	        Status = GetDroppedAppCntrlPktMibs((PVOID)temp_buff,
									(PPER_TARANG_DATA)filp->private_data);
			
			if(copy_to_user((PCHAR)IoBuffer.OutputBuffer,(PCHAR)temp_buff, 
				sizeof(S_MIBS_HOST_STATS_MIBS)))
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Copy to user failed\n");
				bcm_kfree(temp_buff);
				return -EFAULT;
			}
			
			bcm_kfree(temp_buff);
			break;
		}

        case IOCTL_SET_HOST_STATS_MIBS:
		{
			PCHAR temp_buff;
			BCMPayLoadHostMIB *pstRcvPayLOadHostMIB;
			if(copy_from_user((PCHAR)&IoBuffer, 
					(PCHAR)arg, sizeof(IOCTL_BUFFER)))
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Copy_from user for IoBuff failed\n");
				Status = -EFAULT;
				break;
			}
			pstRcvPayLOadHostMIB = (BCMPayLoadHostMIB*)IoBuffer.InputBuffer;

			temp_buff= (PCHAR)kmalloc(pstRcvPayLOadHostMIB->uiLength, GFP_KERNEL);
			if(temp_buff == NULL)
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Out of memory....\n");
				return -ENOMEM;
			}

			if(copy_from_user((PCHAR)temp_buff,pstRcvPayLOadHostMIB->pucRcvBuffer,pstRcvPayLOadHostMIB->uiLength))
			{
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Copy_from user for IoBuff.InputBuffer failed\n");
				Status = -EFAULT;
				break;
			}

			
			Status = ProcessSetHostMibs(Adapter, 
					(PUCHAR)temp_buff, IoBuffer.InputLength);
			
			bcm_kfree(temp_buff);
			break;
		}



		case IOCTL_BCM_WAKE_UP_DEVICE_FROM_IDLE:
			if((FALSE == Adapter->bTriedToWakeUpFromlowPowerMode) && ((TRUE==Adapter->IdleMode)||(TRUE==Adapter->bShutStatus)))
			{

				
				if(Adapter->bShutStatus == TRUE)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK,0,0, "Shutdown Mode !!\n");
					Adapter->usIdleModePattern = ABORT_SHUTDOWN_MODE; 
				}
				else if(Adapter->IdleMode == TRUE)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK,0,0, "Idle Mode !!\n");
					Adapter->usIdleModePattern = ABORT_IDLE_MODE;
				}
				else
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK,0,0, "Invalid Mode Request...!!\n");
					Status = STATUS_FAILURE;
					break;
				}
				
				Adapter->bWakeUpDevice = TRUE;
				wake_up(&Adapter->process_rx_cntrlpkt);
				#if 0
				Adapter->bTriedToWakeUpFromlowPowerMode = TRUE;
				InterfaceAbortIdlemode (Adapter, Adapter->usIdleModePattern);
				#endif
			}
			Status = STATUS_SUCCESS;
			break;

		case IOCTL_BCM_BULK_WRM:
			{
				PBULKWRM_BUFFER pBulkBuffer;
				UINT uiTempVar=0;
				PCHAR pvBuffer = NULL;
				
				if((Adapter->IdleMode == TRUE) || 
					(Adapter->bShutStatus ==TRUE) || 
					(Adapter->bPreparingForLowPowerMode ==TRUE))
				{
                    BCM_DEBUG_PRINT (Adapter, DBG_TYPE_PRINTK, 0, 0, "Device in Idle/Shutdown Mode, Blocking Wrms\n");
					Status = -EACCES;
					break;
				}
				/* Copy Ioctl Buffer structure */
				if(copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, sizeof(IOCTL_BUFFER)))
				{
					Status = -EFAULT;
					break;
				}

				pvBuffer=kmalloc(IoBuffer.InputLength, GFP_KERNEL);
				if(!pvBuffer)
				{
					return -ENOMEM;
					break;
				}

				/* Get WrmBuffer structure */
                if(copy_from_user(pvBuffer, IoBuffer.InputBuffer, IoBuffer.InputLength))
				{
					bcm_kfree(pvBuffer);
					Status = -EFAULT;
					break;
				}

				pBulkBuffer = (PBULKWRM_BUFFER)pvBuffer;

				if(((B_UINT32)pBulkBuffer->Register & 0x0F000000) != 0x0F000000 || 
					((B_UINT32)pBulkBuffer->Register & 0x3))
				{
					bcm_kfree(pvBuffer);
                    BCM_DEBUG_PRINT (Adapter, DBG_TYPE_PRINTK, 0, 0,"WRM Done On invalid Address : %x Access Denied.\n",(int)pBulkBuffer->Register);
					Status = -EINVAL;
					break;
				}

				
				uiTempVar = pBulkBuffer->Register & EEPROM_REJECT_MASK;
				if(!((Adapter->pstargetparams->m_u32Customize)&VSG_MODE) 
				&& 	((uiTempVar == EEPROM_REJECT_REG_1)|| 
						(uiTempVar == EEPROM_REJECT_REG_2) || 
					(uiTempVar == EEPROM_REJECT_REG_3) || 
					(uiTempVar == EEPROM_REJECT_REG_4)) &&
					(cmd == IOCTL_BCM_REGISTER_WRITE))
				{
					bcm_kfree(pvBuffer);
                    BCM_DEBUG_PRINT (Adapter, DBG_TYPE_PRINTK, 0, 0,"EEPROM Access Denied, not in VSG Mode\n");
					Status = -EFAULT;
					break;
				}

				down(&Adapter->rdmWrmDevAccLock); 

				if(pBulkBuffer->SwapEndian == FALSE)
					Status = wrmWithLock(Adapter, (UINT)pBulkBuffer->Register, (PCHAR)pBulkBuffer->Values, IoBuffer.InputLength - 2*sizeof(B_UINT32));
										
				else
					Status = wrmaltWithLock(Adapter, (UINT)pBulkBuffer->Register, (PUINT)pBulkBuffer->Values, IoBuffer.InputLength - 2*sizeof(B_UINT32));

				up(&Adapter->rdmWrmDevAccLock); 
	
				if(Status != STATUS_SUCCESS)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "WRM Failed\n");
				}

				bcm_kfree(pvBuffer);
				break;
			}

		case IOCTL_BCM_GET_NVM_SIZE:
			{
					
			if(copy_from_user((unsigned char *)&IoBuffer, 
					(unsigned char *)arg, sizeof(IOCTL_BUFFER)))
			{
				//IOLog("failed NVM first");
				Status = -EFAULT;
				break;
			}
			if(Adapter->eNVMType == NVM_EEPROM || Adapter->eNVMType == NVM_FLASH ) {
				if(copy_to_user(IoBuffer.OutputBuffer, 
					(unsigned char *)&Adapter->uiNVMDSDSize, (UINT)sizeof(UINT)))
				{
						Status = -EFAULT;
						return Status;
				}
			}
			
			Status = STATUS_SUCCESS ;				
			}
			break;

		case IOCTL_BCM_CAL_INIT :
			{
				UINT uiSectorSize = 0 ;

				if(Adapter->eNVMType == NVM_FLASH)
				{
					Status = copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, sizeof(IOCTL_BUFFER));
					if(Status)
					{
						BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Copy From User space failed. status :%d", Status);
						return Status;
					}
					uiSectorSize = *((PUINT)(IoBuffer.InputBuffer));
					if((uiSectorSize < MIN_SECTOR_SIZE) || (uiSectorSize > MAX_SECTOR_SIZE))
					{
					
						Status = copy_to_user(IoBuffer.OutputBuffer, 
									(unsigned char *)&Adapter->uiSectorSize ,
									(UINT)sizeof(UINT));
						if(Status)
						{
								BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Coping the sector size to use space failed. status:%d",Status);
								return Status;
						}
					}
					else
					{
						if(IsFlash2x(Adapter))
						{
							Status = copy_to_user(IoBuffer.OutputBuffer, 
									(unsigned char *)&Adapter->uiSectorSize ,
									(UINT)sizeof(UINT));
							if(Status)
							{
									BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Coping the sector size to use space failed. status:%d",Status);
									return Status;
							}
							
						}
						else
						{

							down(&Adapter->NVMRdmWrmLock);

							Status = LowPowerModeWakeup(Adapter,ABORT_NVM_ACCESS);
							if(Status != STATUS_SUCCESS)
							{
								BCM_DEBUG_PRINT (Adapter, DBG_TYPE_PRINTK, 0, 0,"Device wakeup failed...nvm access is not allowed\n");
								up(&Adapter->NVMRdmWrmLock);
								return -EACCES;
							}

							Adapter->uiSectorSize = uiSectorSize ;
							BcmUpdateSectorSize(Adapter,Adapter->uiSectorSize);
							up(&Adapter->NVMRdmWrmLock);
						}
					}
					Status = STATUS_SUCCESS ;
				}	
				else
				{
					Status = STATUS_FAILURE;
				}
			}	
			break;	
        case IOCTL_BCM_SET_DEBUG :
            {
                USER_BCM_DBG_STATE sUserDebugState;

//				BCM_DEBUG_PRINT (Adapter, DBG_TYPE_PRINTK, 0, 0, "Entered the ioctl %x \n", IOCTL_BCM_SET_DEBUG );

				BCM_DEBUG_PRINT (Adapter, DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "In SET_DEBUG ioctl\n");
				Status = copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, sizeof(IOCTL_BUFFER));
				if(Status)
				{
					BCM_DEBUG_PRINT (Adapter, DBG_TYPE_PRINTK, 0, 0, "Copy from user failed\n");
					break;
				}
				Status = copy_from_user(&sUserDebugState,(USER_BCM_DBG_STATE *)IoBuffer.InputBuffer, sizeof(USER_BCM_DBG_STATE));
				if(Status)
				{
					BCM_DEBUG_PRINT (Adapter, DBG_TYPE_PRINTK, 0, 0,  "Copy of IoBuffer.InputBuffer failed");
					return Status;
				}

				BCM_DEBUG_PRINT (Adapter, DBG_TYPE_PRINTK, 0, 0, "IOCTL_BCM_SET_DEBUG: OnOff=%d Type = 0x%x ", 
				sUserDebugState.OnOff, sUserDebugState.Type);
				//sUserDebugState.Subtype <<= 1;
				sUserDebugState.Subtype = 1 << sUserDebugState.Subtype;
				BCM_DEBUG_PRINT (Adapter, DBG_TYPE_PRINTK, 0, 0, "actual Subtype=0x%x\n", sUserDebugState.Subtype);

				// Update new 'DebugState' in the Adapter
				Adapter->stDebugState.type |= sUserDebugState.Type;
				/* Subtype: A bitmap of 32 bits for Subtype per Type.
				 * Valid indexes in 'subtype' array: 1,2,4,8
				 * corresponding to valid Type values. Hence we can use the 'Type' field
				 * as the index value, ignoring the array entries 0,3,5,6,7 !
				 */
				if (sUserDebugState.OnOff)
					Adapter->stDebugState.subtype[sUserDebugState.Type] |= sUserDebugState.Subtype;
				else
					Adapter->stDebugState.subtype[sUserDebugState.Type] &= ~sUserDebugState.Subtype;

                BCM_SHOW_DEBUG_BITMAP(Adapter);
               
			}
			break;
		case IOCTL_BCM_NVM_READ:
		case IOCTL_BCM_NVM_WRITE:
			{
				
				NVM_READWRITE  stNVMReadWrite = {0};	
				PUCHAR pReadData = NULL;
				PUCHAR pBuffertobeCopied = NULL;
				B_UINT32 ulDSDMagicNumInUsrBuff = 0 ;
				struct timeval tv0, tv1;
				memset(&tv0,0,sizeof(struct timeval));	
				memset(&tv1,0,sizeof(struct timeval));	
				if((Adapter->eNVMType == NVM_FLASH) && (Adapter->uiFlashLayoutMajorVersion == 0))
				{
					BCM_DEBUG_PRINT(Adapter, DBG_TYPE_PRINTK, 0, 0,"The Flash Control Section is Corrupted. Hence Rejection on NVM Read/Write\n");
					Status = -EFAULT;
					break;
				}

				if(IsFlash2x(Adapter))
				{
					if((Adapter->eActiveDSD != DSD0) &&
						(Adapter->eActiveDSD != DSD1) &&
						(Adapter->eActiveDSD != DSD2))
					{
						BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"No DSD is active..hence NVM Command is blocked");
						return STATUS_FAILURE ;
					}
				}

			/* Copy Ioctl Buffer structure */
			
				if(copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, sizeof(IOCTL_BUFFER)))
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"copy_from_user failed\n");
                    Status = -EFAULT;
					break;
				}
				if(IOCTL_BCM_NVM_READ == cmd)
					pBuffertobeCopied = IoBuffer.OutputBuffer;
				else
					pBuffertobeCopied = IoBuffer.InputBuffer;

				if(copy_from_user(&stNVMReadWrite, pBuffertobeCopied,sizeof(NVM_READWRITE)))
				{
					Status = -EFAULT;
					break;
				}

				//
				// Deny the access if the offset crosses the cal area limit.
				//
				if((stNVMReadWrite.uiOffset + stNVMReadWrite.uiNumBytes) > Adapter->uiNVMDSDSize)
				{
				//BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Can't allow access beyond NVM Size: 0x%x 0x%x\n", stNVMReadWrite.uiOffset ,
//							stNVMReadWrite.uiNumBytes);
					Status = STATUS_FAILURE;
					break;	
				}

				pReadData =(PCHAR)kmalloc(stNVMReadWrite.uiNumBytes, GFP_KERNEL);
			
				if(!pReadData)
					return -ENOMEM;
				
				memset(pReadData,0,stNVMReadWrite.uiNumBytes);

				if(copy_from_user(pReadData, stNVMReadWrite.pBuffer, 
							stNVMReadWrite.uiNumBytes))
				{
					Status = -EFAULT;
					bcm_kfree(pReadData);
					break;
				}
				
				do_gettimeofday(&tv0);	
				if(IOCTL_BCM_NVM_READ == cmd)
				{
					down(&Adapter->NVMRdmWrmLock);
					if(TRUE == Adapter->bUseFlashShmPtr)
					{
						Adapter->bUseFlashShmPtr  = FALSE;
						Adapter->fpFlashBulkRead  = BeceemFlashBulkReadSHM;
						Adapter->fpFlashBulkWrite = BeceemFlashBulkWriteSHM;
						Adapter->bFlashSHMEnabled = TRUE;
					
					}
					
					
					Status = LowPowerModeWakeup(Adapter,ABORT_NVM_ACCESS);
					if(Status != STATUS_SUCCESS)
					{
						BCM_DEBUG_PRINT (Adapter, DBG_TYPE_PRINTK, 0, 0,"Device wakeup failed...nvm access is not allowed\n");
						up(&Adapter->NVMRdmWrmLock);						
						bcm_kfree(pReadData);
						return -EACCES;
					}

					Status = BeceemNVMRead(Adapter, (PUINT)pReadData,
						stNVMReadWrite.uiOffset, stNVMReadWrite.uiNumBytes);
					
					up(&Adapter->NVMRdmWrmLock);

					if(Status != STATUS_SUCCESS)
						{
							bcm_kfree(pReadData);
							return Status;
						}
					if(copy_to_user((PCHAR)stNVMReadWrite.pBuffer, 
							(PCHAR)pReadData, (UINT)stNVMReadWrite.uiNumBytes))
						{
							bcm_kfree(pReadData);
							Status = -EFAULT;
						}
				}
				else
				{
					
					down(&Adapter->NVMRdmWrmLock);
					if(TRUE == Adapter->bUseFlashShmPtr)
					{
						Adapter->bUseFlashShmPtr  = FALSE;
						Adapter->fpFlashBulkRead  = BeceemFlashBulkReadSHM;
						Adapter->fpFlashBulkWrite = BeceemFlashBulkWriteSHM;
						Adapter->bFlashSHMEnabled = TRUE;
					
					}
					
					
					Status = LowPowerModeWakeup(Adapter,ABORT_NVM_ACCESS);
					if(Status != STATUS_SUCCESS)
					{
						BCM_DEBUG_PRINT (Adapter, DBG_TYPE_PRINTK, 0, 0,"Device wakeup failed...nvm access is not allowed\n");
						up(&Adapter->NVMRdmWrmLock);
						bcm_kfree(pReadData);
						return -EACCES;
					}
					
					
					
					Adapter->bHeaderChangeAllowed = TRUE ;
					if(IsFlash2x(Adapter))
					{
						/*
							New Requirement:- 
							DSD section updation will be allowed in two case:-
							1.  if DSD sig is present in DSD header means dongle is ok and updation is fruitfull
							2.  if point 1 failes then user buff should have DSD sig. this point ensures that if dongle is 
							      corrupted then user space program first modify the DSD header with valid DSD sig so 
							      that this as well as further write may be worthwhile.

							 This restriction has been put assuming that if DSD sig is corrupted, DSD 
							 data won't be considered valid.
							
							
						*/
						Status = BcmFlash2xCorruptSig(Adapter,Adapter->eActiveDSD);
						if(Status != STATUS_SUCCESS)
						{
							if(( (stNVMReadWrite.uiOffset + stNVMReadWrite.uiNumBytes) != Adapter->uiNVMDSDSize ) ||
								(stNVMReadWrite.uiNumBytes < SIGNATURE_SIZE))
							{
								BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"DSD Sig is present neither in Flash nor User provided Input..");
								up(&Adapter->NVMRdmWrmLock);
								bcm_kfree(pReadData);
								return Status;
							}
							
							ulDSDMagicNumInUsrBuff = ntohl(*(PUINT)(pReadData + stNVMReadWrite.uiNumBytes - SIGNATURE_SIZE));
							if(ulDSDMagicNumInUsrBuff != DSD_IMAGE_MAGIC_NUMBER)
							{
								BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"DSD Sig is present neither in Flash nor User provided Input..");
								up(&Adapter->NVMRdmWrmLock);
								bcm_kfree(pReadData);
								return Status;
							}
						}
					}
					Status = BeceemNVMWrite(Adapter, (PUINT )pReadData,
									stNVMReadWrite.uiOffset, stNVMReadWrite.uiNumBytes, stNVMReadWrite.bVerify);
					if(IsFlash2x(Adapter))
						BcmFlash2xWriteSig(Adapter,Adapter->eActiveDSD);
					
					Adapter->bHeaderChangeAllowed = FALSE ;

					up(&Adapter->NVMRdmWrmLock);
					
					
					if(Status != STATUS_SUCCESS)
					{
						bcm_kfree(pReadData);
						return Status;
					}
				}
				do_gettimeofday(&tv1);
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, " timetaken by Write/read :%ld msec\n",(tv1.tv_sec - tv0.tv_sec)*1000 +(tv1.tv_usec - tv0.tv_usec)/1000); 	
				
				
				bcm_kfree(pReadData);
				Status = STATUS_SUCCESS;
			}
			break;	
		case IOCTL_BCM_FLASH2X_SECTION_READ :
			 {

				FLASH2X_READWRITE sFlash2xRead = {0}; 
				PUCHAR pReadBuff = NULL ;
				UINT NOB = 0;
				UINT BuffSize = 0;
				UINT ReadBytes = 0;
				UINT ReadOffset = 0;
				PUCHAR OutPutBuff = NULL;

				if(IsFlash2x(Adapter) != TRUE)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Flash Does not have 2.x map");
					return -EINVAL;
				}
				
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "IOCTL_BCM_FLASH2X_SECTION_READ Called");	
				Status = copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, sizeof(IOCTL_BUFFER));
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Copy of IOCTL BUFFER failed");
					return Status ;
				}

				//Reading FLASH 2.x READ structure
				Status = copy_from_user((PUCHAR)&sFlash2xRead, (PUCHAR)IoBuffer.InputBuffer,sizeof(FLASH2X_READWRITE));
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Copy of Input Buffer failed");
					return Status ;
				}
			
								
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"\nsFlash2xRead.Section :%x" ,sFlash2xRead.Section);
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"\nsFlash2xRead.offset :%x" ,sFlash2xRead.offset);
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"\nsFlash2xRead.numOfBytes :%x" ,sFlash2xRead.numOfBytes);
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"\nsFlash2xRead.bVerify :%x\n" ,sFlash2xRead.bVerify);

				//This was internal to driver for raw read. now it has ben exposed to user space app.
				if(validateFlash2xReadWrite(Adapter,&sFlash2xRead) == FALSE)
					return STATUS_FAILURE ;

				NOB = sFlash2xRead.numOfBytes;
				if(NOB > Adapter->uiSectorSize )
					BuffSize = Adapter->uiSectorSize;
				else
					BuffSize = NOB ;
				
				ReadOffset = sFlash2xRead.offset ;
				OutPutBuff = (PUCHAR)(IoBuffer.OutputBuffer) ;

				
				pReadBuff = (PCHAR)kzalloc(BuffSize , GFP_KERNEL);
				if(pReadBuff == NULL)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Memory allocation failed for Flash 2.x Read Structure");
					return -ENOMEM;
				}
				down(&Adapter->NVMRdmWrmLock);

				if(TRUE == Adapter->bUseFlashShmPtr)
				{
					Adapter->bUseFlashShmPtr  = FALSE;
					Adapter->fpFlashBulkRead  = BeceemFlashBulkReadSHM;
					Adapter->fpFlashBulkWrite = BeceemFlashBulkWriteSHM;
					Adapter->bFlashSHMEnabled = TRUE;
					
				}

				Status = LowPowerModeWakeup(Adapter,ABORT_NVM_ACCESS);
				if(Status != STATUS_SUCCESS)
				{
					BCM_DEBUG_PRINT (Adapter, DBG_TYPE_PRINTK, 0, 0,"Device wakeup failed...nvm access is not allowed\n");
					up(&Adapter->NVMRdmWrmLock);
					bcm_kfree(pReadBuff);
					return -EACCES;
				}

				while(NOB)
				{

					if(NOB > Adapter->uiSectorSize )
						ReadBytes = Adapter->uiSectorSize;
					else
						ReadBytes = NOB;
						
					
					//Reading the data from Flash 2.x
					
					Status = BcmFlash2xBulkRead(Adapter,(PUINT)pReadBuff,sFlash2xRead.Section,ReadOffset,ReadBytes);
					if(Status)
					{
						BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"Flash 2x read err with Status :%d", Status);
						break ;
					}

					BCM_DEBUG_PRINT_BUFFER(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,pReadBuff, ReadBytes);

					Status = copy_to_user(OutPutBuff, pReadBuff,ReadBytes);
				 	if(Status)
				 	{
				 		BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"Copy to use failed with status :%d", Status);
						break;
				 	}
					NOB = NOB - ReadBytes;
					if(NOB)
					{
						ReadOffset = ReadOffset + ReadBytes ;
						OutPutBuff = OutPutBuff + ReadBytes ;
					}	
					
				}	
				up(&Adapter->NVMRdmWrmLock);
				bcm_kfree(pReadBuff);
				
			 }
			 break ;	
		case IOCTL_BCM_FLASH2X_SECTION_WRITE :
			 {
			 	FLASH2X_READWRITE sFlash2xWrite = {0};
				PUCHAR pWriteBuff = NULL; 
				PUCHAR InputAddr = NULL;
				UINT NOB = 0;
				UINT BuffSize = 0;
				UINT WriteOffset = 0;
				UINT WriteBytes = 0;
			
				if(IsFlash2x(Adapter) != TRUE)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Flash Does not have 2.x map");
					return -EINVAL;
				}
				
				//First make this False so that we can enable the Sector Permission Check in BeceemFlashBulkWrite
				Adapter->bAllDSDWriteAllow = FALSE;
				
				
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, " IOCTL_BCM_FLASH2X_SECTION_WRITE Called");	
				Status = copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, sizeof(IOCTL_BUFFER));
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Copy of IOCTL BUFFER failed");
					return Status;
				}
								
				//Reading FLASH 2.x READ structure
				Status = copy_from_user((PCHAR)&sFlash2xWrite, (PCHAR)IoBuffer.InputBuffer, sizeof(FLASH2X_READWRITE));
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Reading of output Buffer from IOCTL buffer fails");
					return Status; 
				}
				
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"\nsFlash2xRead.Section :%x" ,sFlash2xWrite.Section);
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"\nsFlash2xRead.offset :%d" ,sFlash2xWrite.offset);
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"\nsFlash2xRead.numOfBytes :%x" ,sFlash2xWrite.numOfBytes);
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"\nsFlash2xRead.bVerify :%x\n" ,sFlash2xWrite.bVerify);
				#if 0
				if((sFlash2xWrite.Section == ISO_IMAGE1) ||(sFlash2xWrite.Section == ISO_IMAGE2) || 
					(sFlash2xWrite.Section == DSD0) || (sFlash2xWrite.Section == DSD1) || (sFlash2xWrite.Section == DSD2))
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"ISO/DSD Image write is not allowed....  ");
					return STATUS_FAILURE ;
				}
				#endif
				if((sFlash2xWrite.Section != VSA0) && (sFlash2xWrite.Section != VSA1) && 
					(sFlash2xWrite.Section != VSA2) )
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"Only VSA write is allowed");
					return -EINVAL;
				}
				
				if(validateFlash2xReadWrite(Adapter,&sFlash2xWrite) == FALSE)
					return STATUS_FAILURE ;
				
				InputAddr = (PCHAR)(sFlash2xWrite.pDataBuff) ;
				WriteOffset = sFlash2xWrite.offset ;
				NOB = sFlash2xWrite.numOfBytes;

				if(NOB > Adapter->uiSectorSize )
					BuffSize = Adapter->uiSectorSize;
				else
					BuffSize = NOB ;
				
				pWriteBuff = (PCHAR)kmalloc(BuffSize, GFP_KERNEL);
				if(pWriteBuff == NULL)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Memory allocation failed for Flash 2.x Read Structure");
					return -ENOMEM;
				}
				
				//extracting the remainder of the given offset.
				WriteBytes = Adapter->uiSectorSize ;
				if(WriteOffset % Adapter->uiSectorSize)
					WriteBytes =Adapter->uiSectorSize - (WriteOffset % Adapter->uiSectorSize);
				if(NOB < WriteBytes)
					WriteBytes = NOB;
				
				down(&Adapter->NVMRdmWrmLock);

				if(TRUE == Adapter->bUseFlashShmPtr)
				{
					Adapter->bUseFlashShmPtr  = FALSE;
					Adapter->fpFlashBulkRead  = BeceemFlashBulkReadSHM;
					Adapter->fpFlashBulkWrite = BeceemFlashBulkWriteSHM;
					Adapter->bFlashSHMEnabled = TRUE;
					
				}

				Status = LowPowerModeWakeup(Adapter,ABORT_NVM_ACCESS);
				if(Status != STATUS_SUCCESS)
				{
					BCM_DEBUG_PRINT (Adapter, DBG_TYPE_PRINTK, 0, 0,"Device wakeup failed...nvm access is not allowed\n");
					up(&Adapter->NVMRdmWrmLock);
					bcm_kfree(pWriteBuff);
					return -EACCES;
				}

				BcmFlash2xCorruptSig(Adapter,sFlash2xWrite.Section);	
				do
				{
					Status = copy_from_user(pWriteBuff,InputAddr,WriteBytes);
				 	if(Status)
				 	{
				 		BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Copy to user failed with status :%d", Status);
						break ;
				 	}
					BCM_DEBUG_PRINT_BUFFER(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,pWriteBuff,WriteBytes);	
					//Writing the data from Flash 2.x
					Status = BcmFlash2xBulkWrite(Adapter,(PUINT)pWriteBuff,sFlash2xWrite.Section,WriteOffset,WriteBytes,sFlash2xWrite.bVerify);
					
					if(Status)
					{
						BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Flash 2x read err with Status :%d", Status);
						break ;
					}

					NOB = NOB - WriteBytes;
					if(NOB)
					{
						WriteOffset = WriteOffset + WriteBytes ;
						InputAddr = InputAddr + WriteBytes ;
						if(NOB > Adapter->uiSectorSize )
							WriteBytes = Adapter->uiSectorSize;
						else
							WriteBytes = NOB;
					}
					

				}	while(NOB > 0);
				BcmFlash2xWriteSig(Adapter,sFlash2xWrite.Section);
				up(&Adapter->NVMRdmWrmLock);
				bcm_kfree(pWriteBuff);
			 }
			 break ;
		case IOCTL_BCM_GET_FLASH2X_SECTION_BITMAP :
			 {
				
			 	PFLASH2X_BITMAP psFlash2xBitMap = NULL ;
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "IOCTL_BCM_GET_FLASH2X_SECTION_BITMAP Called");	

				Status = copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, sizeof(IOCTL_BUFFER));
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Copy of IOCTL BUFFER failed");
					return Status;
				}
				if(IoBuffer.OutputLength != sizeof(FLASH2X_BITMAP))
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Structure size mismatch Lib :0x%x Driver :0x%lx ",IoBuffer.OutputLength,(ULONG)(sizeof(FLASH2X_BITMAP)));
					break;					
				}
				
				psFlash2xBitMap = (PFLASH2X_BITMAP)kzalloc(sizeof(FLASH2X_BITMAP), GFP_KERNEL);
				if(psFlash2xBitMap == NULL)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Memory is not available");
					return -ENOMEM ;
				}	
				//Reading the Flash Sectio Bit map
				down(&Adapter->NVMRdmWrmLock);

				if(TRUE == Adapter->bUseFlashShmPtr)
				{
					Adapter->bUseFlashShmPtr  = FALSE;
					Adapter->fpFlashBulkRead  = BeceemFlashBulkReadSHM;
					Adapter->fpFlashBulkWrite = BeceemFlashBulkWriteSHM;
					Adapter->bFlashSHMEnabled = TRUE;
					
				}
				
				Status = LowPowerModeWakeup(Adapter,ABORT_NVM_ACCESS);
				if(Status != STATUS_SUCCESS)
				{
					BCM_DEBUG_PRINT (Adapter, DBG_TYPE_PRINTK, 0, 0,"Device wakeup failed...nvm access is not allowed\n");
					up(&Adapter->NVMRdmWrmLock);
					bcm_kfree(psFlash2xBitMap);
					return -EACCES;
				}
				
				
				BcmGetFlash2xSectionalBitMap(Adapter, psFlash2xBitMap);
				up(&Adapter->NVMRdmWrmLock);
				Status = copy_to_user((PCHAR)IoBuffer.OutputBuffer, (PCHAR)psFlash2xBitMap, sizeof(FLASH2X_BITMAP));
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "copying Flash2x bitMap failed");
					bcm_kfree(psFlash2xBitMap);
					return Status; 
				}
				bcm_kfree(psFlash2xBitMap);
			 }
			 break ;
		case IOCTL_BCM_SET_ACTIVE_SECTION :
			 {
				FLASH2X_SECTION_VAL eFlash2xSectionVal = 0;
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "IOCTL_BCM_SET_ACTIVE_SECTION Called");

				if(IsFlash2x(Adapter) != TRUE)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Flash Does not have 2.x map");
					return -EINVAL;
				}
					
				Status = copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, sizeof(IOCTL_BUFFER));
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Copy of IOCTL BUFFER failed");
					return Status;
				}

				Status = copy_from_user((PCHAR)&eFlash2xSectionVal,(PCHAR)IoBuffer.InputBuffer, sizeof(INT));
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Copy of flash section val failed");
					return Status;
				}

				down(&Adapter->NVMRdmWrmLock);

				if(TRUE == Adapter->bUseFlashShmPtr)
				{
					Adapter->bUseFlashShmPtr  = FALSE;
					Adapter->fpFlashBulkRead  = BeceemFlashBulkReadSHM;
					Adapter->fpFlashBulkWrite = BeceemFlashBulkWriteSHM;
					Adapter->bFlashSHMEnabled = TRUE;
					
				}

				Status = LowPowerModeWakeup(Adapter,ABORT_NVM_ACCESS);
				if(Status != STATUS_SUCCESS)
				{
					BCM_DEBUG_PRINT (Adapter, DBG_TYPE_PRINTK, 0, 0,"Device wakeup failed...nvm access is not allowed\n");
					up(&Adapter->NVMRdmWrmLock);
					return -EACCES;
				}

				Status = BcmSetActiveSection(Adapter,eFlash2xSectionVal);
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Failed to make it's priority Highest. Status %d", Status);
				}
				up(&Adapter->NVMRdmWrmLock);
			}
			break ;
		case IOCTL_BCM_IDENTIFY_ACTIVE_SECTION :
			 {
			 	//Right Now we are taking care of only DSD
				Adapter->bAllDSDWriteAllow = FALSE ;
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"IOCTL_BCM_IDENTIFY_ACTIVE_SECTION called");

				#if 0
				SECTION_TYPE section = 0 ;
				
				
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "IOCTL_BCM_IDENTIFY_ACTIVE_SECTION Called");	
				Status = copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, sizeof(IOCTL_BUFFER));
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "Copy of IOCTL BUFFER failed");
					return Status;
				}
				Status = copy_from_user((PCHAR)section,(PCHAR)&IoBuffer, sizeof(INT));
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "Copy of section type failed failed");
					return Status;
				}
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"Read Section :%d", section);
			 	if(section == DSD)
					Adapter->ulFlashCalStart = Adapter->uiActiveDSDOffsetAtFwDld ;
				else
					Status = STATUS_FAILURE ;
				#endif
				Status = STATUS_SUCCESS ;
			 }
			 break ;
		case IOCTL_BCM_COPY_SECTION :
			 {
				FLASH2X_COPY_SECTION sCopySectStrut = {0};
				Status = STATUS_SUCCESS;
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "IOCTL_BCM_COPY_SECTION  Called");

				Adapter->bAllDSDWriteAllow = FALSE ;
				
				if(IsFlash2x(Adapter) != TRUE)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Flash Does not have 2.x map");
					return -EINVAL;
				}
							
				Status = copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, sizeof(IOCTL_BUFFER));
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Copy of IOCTL BUFFER failed Status :%d", Status);
					return Status;
				}

				Status = copy_from_user((PCHAR)&sCopySectStrut,(PCHAR)IoBuffer.InputBuffer, sizeof(FLASH2X_COPY_SECTION));
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Copy of Copy_Section_Struct failed with Status :%d", Status);
					return Status;
				}
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "Source SEction :%x", sCopySectStrut.SrcSection);
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "Destination SEction :%x", sCopySectStrut.DstSection);
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "offset :%x", sCopySectStrut.offset);
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "NOB :%x", sCopySectStrut.numOfBytes);
				
				
				if(IsSectionExistInFlash(Adapter,sCopySectStrut.SrcSection) == FALSE) 
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Source Section<%x> does not exixt in Flash ", sCopySectStrut.SrcSection);
					return -EINVAL;
				}

				if(IsSectionExistInFlash(Adapter,sCopySectStrut.DstSection) == FALSE) 
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Destinatio Section<%x> does not exixt in Flash ", sCopySectStrut.DstSection);
					return -EINVAL;
				}

				if(sCopySectStrut.SrcSection == sCopySectStrut.DstSection)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"Source and Destination section should be different");
					return -EINVAL;
				}

				down(&Adapter->NVMRdmWrmLock);

				if(TRUE == Adapter->bUseFlashShmPtr)
				{
					Adapter->bUseFlashShmPtr  = FALSE;
					Adapter->fpFlashBulkRead  = BeceemFlashBulkReadSHM;
					Adapter->fpFlashBulkWrite = BeceemFlashBulkWriteSHM;
					Adapter->bFlashSHMEnabled = TRUE;
					
				}
				
				Status = LowPowerModeWakeup(Adapter,ABORT_NVM_ACCESS);
				if(Status != STATUS_SUCCESS)
				{
					BCM_DEBUG_PRINT (Adapter, DBG_TYPE_PRINTK, 0, 0,"Device wakeup failed...nvm access is not allowed\n");
					up(&Adapter->NVMRdmWrmLock);
					return -EACCES;
				}

				if(sCopySectStrut.SrcSection == ISO_IMAGE1 || sCopySectStrut.SrcSection == ISO_IMAGE2)
				{
					if(IsNonCDLessDevice(Adapter))
					{
						BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Device is Non-CDLess hence won't have ISO !!");
						Status = -EINVAL ;
					}
					else if(sCopySectStrut.numOfBytes == 0)
					{
						Status = BcmCopyISO(Adapter,sCopySectStrut);	
					}
					else
					{
						BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Partial Copy of ISO section is not Allowed..");
						Status = STATUS_FAILURE ;
					}
					up(&Adapter->NVMRdmWrmLock);
					return Status;
				}

				Status = BcmCopySection(Adapter, sCopySectStrut.SrcSection,
							sCopySectStrut.DstSection,sCopySectStrut.offset,sCopySectStrut.numOfBytes);
				up(&Adapter->NVMRdmWrmLock);
			 }
			 break ;
		case IOCTL_BCM_GET_FLASH_CS_INFO :
			 {
				Status = STATUS_SUCCESS;
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, " IOCTL_BCM_GET_FLASH_CS_INFO Called");	

				Status = copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, sizeof(IOCTL_BUFFER));
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Copy of IOCTL BUFFER failed");
					break;
				}
				if(Adapter->eNVMType != NVM_FLASH)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Connected device does not have flash");
					Status = -EINVAL;
					break;
				}

				if(IsFlash2x(Adapter) == TRUE)
				{
									
					if(IoBuffer.OutputLength < sizeof(FLASH2X_CS_INFO))
					{
						BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0," Passed buffer size:0x%X is insufficient for the CS structure.. \nRequired size :0x%lx ",IoBuffer.OutputLength,(ULONG)(sizeof(FLASH2X_CS_INFO)));
						Status = -EINVAL;
						break;					
					}
				
					Status = copy_to_user((PCHAR)IoBuffer.OutputBuffer, (PCHAR)Adapter->psFlash2xCSInfo, sizeof(FLASH2X_CS_INFO));
					if(Status)
					{
						BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "copying Flash2x cs info failed");
						break; 
					}
				}
				else
				{
					if(IoBuffer.OutputLength < sizeof(FLASH_CS_INFO))
					{
						BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0," Passed buffer size:0x%X is insufficient for the CS structure.. Required size :0x%lu ",IoBuffer.OutputLength,(ULONG)(sizeof(FLASH_CS_INFO)));
						Status = -EINVAL;
						break;					
					}
					Status = copy_to_user((PCHAR)IoBuffer.OutputBuffer, (PCHAR)Adapter->psFlashCSInfo, sizeof(FLASH_CS_INFO));
					if(Status)
					{
						BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "copying Flash CS info failed");
						break; 
					}
				
			 	 }
			  }
			  break ;
		case IOCTL_BCM_SELECT_DSD :
			 {
				UINT SectOfset = 0;
				FLASH2X_SECTION_VAL eFlash2xSectionVal;
				eFlash2xSectionVal = NO_SECTION_VAL ;	
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, " IOCTL_BCM_SELECT_DSD Called");

				if(IsFlash2x(Adapter) != TRUE)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Flash Does not have 2.x map");
					return -EINVAL;
				}
				
				Status = copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, sizeof(IOCTL_BUFFER));
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Copy of IOCTL BUFFER failed");
					return Status;
				}
				Status = copy_from_user((PCHAR)&eFlash2xSectionVal,(PCHAR)IoBuffer.InputBuffer, sizeof(INT));
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Copy of flash section val failed");
					return Status;
				}
				
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"Read Section :%d", eFlash2xSectionVal);
				if((eFlash2xSectionVal != DSD0) && 
					(eFlash2xSectionVal != DSD1) &&
					(eFlash2xSectionVal != DSD2) )
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Passed section<%x> is not DSD section", eFlash2xSectionVal);
					return STATUS_FAILURE ;
				}
				
				SectOfset= BcmGetSectionValStartOffset(Adapter,eFlash2xSectionVal);
				if(SectOfset == INVALID_OFFSET)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Provided Section val <%d> does not exixt in Flash 2.x", eFlash2xSectionVal);
					return -EINVAL;
				}

				Adapter->bAllDSDWriteAllow = TRUE ;

				Adapter->ulFlashCalStart = SectOfset ;
				Adapter->eActiveDSD = eFlash2xSectionVal;
			 }
			 Status = STATUS_SUCCESS ;
			 break;
			 
		case IOCTL_BCM_NVM_RAW_READ :
			 {

				NVM_READWRITE  stNVMRead = {0};	
				INT NOB ;
				INT BuffSize ;
				INT ReadOffset = 0;
				UINT ReadBytes = 0 ;
				PUCHAR pReadBuff = NULL ;
				PUCHAR OutPutBuff = NULL ;

				if(Adapter->eNVMType != NVM_FLASH)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"NVM TYPE is not Flash ");
					return -EINVAL ;	
				}
				
				/* Copy Ioctl Buffer structure */
				if(copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, sizeof(IOCTL_BUFFER)))
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "copy_from_user 1 failed\n");
					Status = -EFAULT;
					break;
				}

				if(copy_from_user(&stNVMRead, (PUCHAR)IoBuffer.OutputBuffer,sizeof(NVM_READWRITE)))
				{
					Status = -EFAULT;
					break;
				}
			
				NOB = stNVMRead.uiNumBytes;
				//In Raw-Read max Buff size : 64MB 
				
				if(NOB > DEFAULT_BUFF_SIZE)
					BuffSize = DEFAULT_BUFF_SIZE;
				else
					BuffSize = NOB ;
				
				ReadOffset = stNVMRead.uiOffset ;
				OutPutBuff = (PUCHAR)(stNVMRead.pBuffer) ;

				
				pReadBuff = (PCHAR)kzalloc(BuffSize , GFP_KERNEL);
				if(pReadBuff == NULL)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Memory allocation failed for Flash 2.x Read Structure");
					Status = -ENOMEM;
					break;
				}
				down(&Adapter->NVMRdmWrmLock);

				if(TRUE == Adapter->bUseFlashShmPtr)
				{
					Adapter->bUseFlashShmPtr  = FALSE;
					Adapter->fpFlashBulkRead  = BeceemFlashBulkReadSHM;
					Adapter->fpFlashBulkWrite = BeceemFlashBulkWriteSHM;
					Adapter->bFlashSHMEnabled = TRUE;
					
				}
				

				Status = LowPowerModeWakeup(Adapter,ABORT_NVM_ACCESS);
				if(Status != STATUS_SUCCESS)
				{
					BCM_DEBUG_PRINT (Adapter, DBG_TYPE_PRINTK, 0, 0,"Device wakeup failed...nvm access is not allowed\n");
					bcm_kfree(pReadBuff);	
					up(&Adapter->NVMRdmWrmLock);
					return -EACCES;
				}
				
				
				Adapter->bFlashRawRead = TRUE ;
				while(NOB)
				{
					if(NOB > DEFAULT_BUFF_SIZE )
						ReadBytes = DEFAULT_BUFF_SIZE;
					else
						ReadBytes = NOB;
						
					//Reading the data from Flash 2.x
					Status = BeceemNVMRead(Adapter,(PUINT)pReadBuff,ReadOffset,ReadBytes);
					if(Status)
					{
						BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Flash 2x read err with Status :%d", Status);
						break;
					}

					BCM_DEBUG_PRINT_BUFFER(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,pReadBuff, ReadBytes);

					Status = copy_to_user(OutPutBuff, pReadBuff,ReadBytes);
				 	if(Status)
				 	{
				 		BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"Copy to use failed with status :%d", Status);
						break;
				 	}
					NOB = NOB - ReadBytes;
					if(NOB)
					{
						ReadOffset = ReadOffset + ReadBytes ;
						OutPutBuff = OutPutBuff + ReadBytes ;
					}	
					
				}	
				Adapter->bFlashRawRead = FALSE ;
				up(&Adapter->NVMRdmWrmLock);
				bcm_kfree(pReadBuff);	
				break ;
			 }

		case IOCTL_BCM_CNTRLMSG_MASK:
			 {
				B_UINT32 RxCntrlMsgBitMask = 0 ;
				
				/* Copy Ioctl Buffer structure */
				Status = copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, sizeof(IOCTL_BUFFER));
				if(Status)
				{   
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"copy of Ioctl buffer is failed from user space");
					break;
				}
				
				Status = copy_from_user(&RxCntrlMsgBitMask, IoBuffer.InputBuffer, IoBuffer.InputLength);
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"copy of control bit mask failed from user space");
					break;
				}
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"\n Got user defined cntrl msg bit mask :%x", RxCntrlMsgBitMask);
				pTarang->RxCntrlMsgBitMask = RxCntrlMsgBitMask ;
			 }
			 break;
			case IOCTL_BCM_GET_DEVICE_DRIVER_INFO:
			{
				DEVICE_DRIVER_INFO DevInfo;

				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"Called IOCTL_BCM_GET_DEVICE_DRIVER_INFO\n");

                memset(&DevInfo,0,sizeof(DEVICE_DRIVER_INFO));

				DevInfo.MaxRDMBufferSize = BUFFER_4K;
				DevInfo.u32DSDStartOffset = EEPROM_CALPARAM_START;
				DevInfo.u32RxAlignmentCorrection = 0; 
				DevInfo.u32NVMType = Adapter->eNVMType;


                if(Adapter->fw_download_done == TRUE)                
                {
                    DevInfo.u32DeviceStatus = DEV_NOT_IN_RESET;
                }
                else
                {
                    DevInfo.u32DeviceStatus = DEV_IN_RESET;
                }

#ifndef BCM_SHM_INTERFACE
				DevInfo.u32InterfaceType = BCM_USB;

#else
				DevInfo.u32InterfaceType = BCM_SHM;
#endif
				Status = copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, sizeof(IOCTL_BUFFER));
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Copy of IOCTL BUFFER failed");
					break;
				}
				if(IoBuffer.OutputLength < sizeof(DevInfo))
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"User Passed buffer length is less than actural buffer size");
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"user passed buffer size :0x%X, expected size :0x%lx",IoBuffer.OutputLength,(ULONG) (sizeof(DevInfo)));
					Status = -EINVAL;
					break;
				}
				Status = copy_to_user((PCHAR)IoBuffer.OutputBuffer, (PCHAR)&DevInfo, sizeof(DevInfo));
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"copying Dev info structure to user space buffer failed");
					break; 
				}
			}
			break ;

			case IOCTL_BCM_TIME_SINCE_NET_ENTRY:
			{
				ST_TIME_ELAPSED stTimeElapsedSinceNetEntry = {0};
				struct timeval tv = {0} ;

				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL,"IOCTL_BCM_TIME_SINCE_NET_ENTRY called");
				
				Status = copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, sizeof(IOCTL_BUFFER));
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Copy of IOCTL BUFFER failed");
					break;
				}
				if(IoBuffer.OutputLength < sizeof(ST_TIME_ELAPSED))
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"User Passed buffer length:0x%x is less than expected buff size :0x%lX",IoBuffer.OutputLength,(ULONG)(sizeof(ST_TIME_ELAPSED)));
					Status = -EINVAL;
					break;
				}
				
				//stTimeElapsedSinceNetEntry.ul64TimeElapsedSinceNetEntry = Adapter->liTimeSinceLastNetEntry;
				do_gettimeofday(&tv);
				stTimeElapsedSinceNetEntry.ul64TimeElapsedSinceNetEntry = tv.tv_sec - Adapter->liTimeSinceLastNetEntry;
				
				Status = copy_to_user((PCHAR)IoBuffer.OutputBuffer, (PCHAR)&stTimeElapsedSinceNetEntry, sizeof(ST_TIME_ELAPSED));
				if(Status)
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"copying ST_TIME_ELAPSED structure to user space buffer failed");
					break; 
				}
				
			}
			break;

#ifdef BCM_SHM_INTERFACE
			case IOCTL_BCM_CONFIG_FILE_DOWNLOAD:
			{
			
				FIRMWARE_INFO sFwInfo;

				if(down_trylock(&Adapter->fw_download_sema))
				{
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Unable to acquire fw_download_sema semaphore!!!\n");
					Status=-EINVAL;
					break;
				}

			
				if(copy_from_user((PCHAR)&IoBuffer, (PCHAR)arg, 
						sizeof(IOCTL_BUFFER)))
				{
					
					up(&Adapter->fw_download_sema);
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "copy_from_user 1 failed\n");
					Status = -EFAULT;
					break;
				}
				BCM_DEBUG_PRINT(Adapter,DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "Length for CFG DLD is : %x\n", 
										IoBuffer.InputLength);
				
				
				
				if(! IoBuffer.InputBuffer || (IoBuffer.InputLength== 0))
				{
					
					up(&Adapter->fw_download_sema);
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "Something else is wrong %u\n", 
					IoBuffer.InputLength);
					Status = -EINVAL;
					break;
				}
				
				
				
				sFwInfo.pvMappedFirmwareAddress	= (PVOID)IoBuffer.InputBuffer;
				
				sFwInfo.u32FirmwareLength	= IoBuffer.InputLength;
				sFwInfo.u32StartingAddress	= (B_UINT32)CONFIG_BEGIN_ADDR;

				Adapter->waiting_to_fw_download_done = FALSE;
				
				Status = bcm_ioctl_cfg_download(Adapter, &sFwInfo );	
				if(Status != STATUS_SUCCESS)
				{
					up(&Adapter->fw_download_sema);
					break;
				}

				if(Adapter->fw_download_done==FALSE)
				{
					
					Status = wait_event_interruptible_timeout(Adapter->ioctl_fw_dnld_wait_queue, 
						Adapter->waiting_to_fw_download_done, msecs_to_jiffies(2000));
	
					if(Status == -ERESTARTSYS)
					{
						BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"\n Asking to exit ..\n");	
					}
					if(Status == 0 || Adapter->waiting_to_fw_download_done == FALSE )
					{
						BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0,"\n Time out happens or flag is false ..\n");	
					}
					BCM_DEBUG_PRINT(Adapter,DBG_TYPE_PRINTK, 0, 0, "\n Firmware Downloaded \n");
					Adapter->fw_download_done=TRUE;
					Status = STATUS_SUCCESS;
				}
				
				up(&Adapter->fw_download_sema);
				break;	
			}
#endif


		default: 
            BCM_DEBUG_PRINT (Adapter, DBG_TYPE_PRINTK, 0, 0, "wrong input %x",cmd);
			BCM_DEBUG_PRINT (Adapter, DBG_TYPE_OTHERS, OSAL_DBG, DBG_LVL_ALL, "In default ioctl %d\n", cmd);
			 Status = STATUS_FAILURE;

			break;
	}
	return Status;
}


#if defined(HAVE_UNLOCKED_IOCTL)
static long bcm_char_unlocked_ioctl(struct file *filp, UINT cmd, ULONG arg)
{
	return (long) bcm_char_ioctl(NULL, filp, cmd, arg);
}
#elif defined(HAVE_COMPAT_IOCTL)
static long bcm_char_compat_ioctl(struct file *filp, UINT cmd, ULONG arg)
{
	return (long) bcm_char_ioctl(NULL, filp, cmd, arg);
}
#endif


static struct file_operations bcm_fops = {
    .owner    = THIS_MODULE,
    .open     = bcm_char_open,
    .release  = bcm_char_release,
    .read     = bcm_char_read,
#if defined(HAVE_UNLOCKED_IOCTL)
    .unlocked_ioctl   =  bcm_char_unlocked_ioctl,
#elif defined(HAVE_COMPAT_IOCTL)
    .compat_ioctl   =  bcm_char_compat_ioctl,
#else
    .ioctl    = bcm_char_ioctl,
#endif
};


int register_control_device_interface(PMINI_ADAPTER Adapter)
{
	if(Adapter->major>0)
    	return Adapter->major;
    Adapter->major = register_chrdev(0, "tarang", &bcm_fops);
    if(Adapter->major < 0)
    {
    	BCM_DEBUG_PRINT(Adapter,DBG_TYPE_INITEXIT, DRV_ENTRY, DBG_LVL_ALL, "register_chrdev:Failed to registering WiMax control char device!");
        return Adapter->major;
    }

	bcm_class = NULL;
	bcm_class = class_create (THIS_MODULE, "tarang");
	if(IS_ERR (bcm_class))
	{
    	BCM_DEBUG_PRINT(Adapter,DBG_TYPE_INITEXIT, DRV_ENTRY, DBG_LVL_ALL, "Unable to create class\n");
        unregister_chrdev(Adapter->major, "tarang");
		Adapter->major = 0;
		return -ENODEV;
	}
	Adapter->pstCreatedClassDevice = device_create (bcm_class, NULL, 
								MKDEV(Adapter->major, 0), 
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,26)
								NULL	,
#endif
								"tarang");

	if(IS_ERR(Adapter->pstCreatedClassDevice))
	{
		BCM_DEBUG_PRINT(Adapter,DBG_TYPE_INITEXIT, DRV_ENTRY, DBG_LVL_ALL, "class device did not get created : %ld", PTR_ERR(Adapter->pstCreatedClassDevice) );
	}
	BCM_DEBUG_PRINT(Adapter,DBG_TYPE_INITEXIT, DRV_ENTRY, DBG_LVL_ALL, "Got Major No: %d", Adapter->major);
    return 0;
}

void unregister_control_device_interface(PMINI_ADAPTER Adapter)
{
	if(Adapter->major > 0)
	{
        BCM_DEBUG_PRINT(Adapter,DBG_TYPE_INITEXIT, DRV_ENTRY, DBG_LVL_ALL, "destroying class device");
		device_destroy (bcm_class, MKDEV(Adapter->major, 0));
	}
    if(!IS_ERR(bcm_class))
	{
        BCM_DEBUG_PRINT(Adapter,DBG_TYPE_INITEXIT, DRV_ENTRY, DBG_LVL_ALL, "destroying created class ");
        class_destroy (bcm_class);
		bcm_class = NULL;
	}
	if(Adapter->major > 0)
	{
		BCM_DEBUG_PRINT(Adapter,DBG_TYPE_INITEXIT, DRV_ENTRY, DBG_LVL_ALL,"unregistering character interface");
        unregister_chrdev(Adapter->major, "tarang");
	}

}
